const { hashSync, genSaltSync } = require("bcryptjs");
const path = require("path");
const { Op } = require("sequelize");
const User = require("../models/UsersModel.js");
const Feature = require("../models/FeaturesModel.js");
const Image = require("../models/ImagesModel.js");
const ExtraFeats = require("../models/ExtraFeatsModel.js");
const Profiles = require("../models/ProfilesModel.js");
const Nationality = require("../models/Nationality.js");
const ProjectsTalents = require("../models/ProjectsTalentsModel");
const Projects = require("../models/ProjectsModel");
const fs = require("fs");
const { copyFile, rename } = require("fs/promises");
const Filters = require("../models/Filters.js");
const ShortList = require("../models/ShortList.js");
const ExtraFilters = require("../models/ExtraFilters.js");
const ClientShare = require("../models/ClientShare.js");
const ClientsRoutes = require("../models/ClientsRoutes.js");
const ClientsModel = require("../models/ClientsModel.js");
const rootPath = "api";
const directoryPathProfile = `../${rootPath}/public/images/profiles/`;
const directoryPathTalents = `../${rootPath}/public/images/talents/`;
const directoryPathClients = `../${rootPath}/public/images/clients/`;
const sharp = require("sharp");
const crypto = require("crypto");
const { json, where } = require("sequelize");
let cryptoCode = "";
const fileExtension = "webp";
const { v4: uuidv4 } = require("uuid");
const sequelize = require("../config/dbseq.js");

/*************************************************************/
/* ---------- ENCODE IMAGE FILENAME  ----------------------- */
/*********************************************************** */
const cryptoFile = (cryptoCode) => {
  cryptoCode =
    //crypto.randomBytes(20).toString("hex") + cryptoCode.replace(/\..+$/, "");
    crypto.randomBytes(20).toString("hex") +
    cryptoCode.substring(cryptoCode.indexOf(".") - 1);
  return cryptoCode;
};

/*************************************************************/
/* ---------- SHARP PROCESSING ---------------------------- */
/*********************************************************** */
const sharpProcess = async (fileBuffer, destinationPath, cryptoCode) => {
  return new Promise((resolve, reject) => {
    sharp(fileBuffer, { failOnError: false })
      .withMetadata()
      .resize({ height: 1000, withoutEnlargement: true })
      .rotate()
      .toFormat("webp")
      .toFile(
        path.join(
          destinationPath,
          `${cryptoCode.replace(/\.[^.]+$/, "")}.webp`
        ),
        (err, info) => {
          if (err) {
            reject(err); // Reject promise if there's an error
          } else {
            resolve({ filename: cryptoCode, info: info }); // Resolve with processed image info
          }
        }
      );
  });
};

/*************************************************************/
/* ---------- PROCESSING PROFILE IMAGE --------------------- */
/*********************************************************** */
const processImage = async (file, destinationPath) => {
  try {
    const fileBuffer = file[0].buffer; // Assuming there's only one file in the array
    const originalName = file[0].originalname; // Assuming there's only one file in the array
    const cryptoCode = cryptoFile(originalName.toLowerCase());
    const processedImage = await sharpProcess(
      fileBuffer,
      destinationPath,
      cryptoCode
    );
    return processedImage;
  } catch (error) {
    console.error("Error processing image:", error);
    throw error;
  }
};

/*************************************************************/
/* ---------- PROCCESSING GALLERY IMAGES ------------------- */
/*********************************************************** */
const processImages = async (files, destinationPath) => {
  // console.log(files, destinationPath);
  try {
    const processedImages = await Promise.all(
      files.map(async (get) => {
        cryptoCode = cryptoFile(get.originalname);
        //      console.log(cryptoCode);
        try {
          let processedImage = await sharpProcess(
            get.buffer,
            destinationPath,
            cryptoCode
          );
          return processedImage; // Return the result of the processing
        } catch (error) {
          console.error(error);
          throw error; // Re-throw the error if something goes wrong
        }
      })
    );
    return processedImages;
  } catch (error) {
    console.error(error);
    throw error;
  }
};

/*************************************************************/
/* ---------- CROPPER  ------------------------------------- */
/*********************************************************** */
/* Sequelize - Add/Update Cropper Image */
const seqCropper = async (req, res) => {
  const userId = req.params.id;
  let fileName = "";
  const originalName = req.body.fileName + ".webp";

  try {
    if (req.files) {
      req.files.uploadImage !== undefined &&
        Object.values(req.files.uploadImage).map((get) => {
          fileName = get.filename;
        });
    }

    if (req.files.uploadImage !== undefined) {
      await copyFile(
        path.join(directoryPathProfile) + `${fileName}`,
        path.join(directoryPathTalents) + `${fileName}`
      );
      /*  fs.unlink(directoryPathProfile + `${fileName}`, (err) => {
        console.log("Profile image deleted !");
      });*/
      if (fileName) {
        Image.create({
          gid: userId,
          image: fileName,
          userId: userId, // Associate with userID
        });
        /* Update User Data */
        User.update({ profileimage: fileName }, { where: { id: userId } }).then(
          () => res.send({ response: "User Updated" })
        );
      }
    }
  } catch (error) {
    if (error) console.log(error);
    res.status(400).json({ response: "Not OK" });
  }
};

// Function to delete a file
const deleteFile = (filePath) => {
  fs.unlink(filePath, (err) => {
    if (err) {
      console.error(`Error deleting file: ${err}`);
    } else {
      console.log(`Unused File deleted successfully: ${filePath}`);
    }
  });
};

// Function to remove an entry from the database
const removeFromDatabase = async (image) => {
  try {
    await Image.destroy({ where: { image: image } });
    console.log(`Entry removed from database: ${image}`);
  } catch (error) {
    console.error(`Error removing entry from database: ${error}`);
  }
};

/*************************************************************/
/* ---------- DELETE USER ---------------------------------- */
/*********************************************************** */
/* Sequelize - Delete User */
const seqRemove = async (req, res) => {
  /* Set Variables */
  const id = req.params.id;
  const filesName = req.body.imageData;
  /* Delete User/Features */
  return User.destroy({
    where: {
      id: id,
    },
  })
    .then(() => {
      /* Delete User Galleries from DB*/
      return Image.destroy({
        where: {
          gid: id,
        },
      }).then(() => {
        /* Delete User files */
        if (filesName) {
          {
            filesName
              .filter((filter) => filter.gid == id)
              .map((del) => {
                console.log(del);
                fs.unlink(directoryPathTalents + del.image, (err) => {
                  console.log("Talents images deleted !");
                });
                fs.unlink(directoryPathProfile + del.image, (err) => {
                  console.log("Profile image deleted !");
                });
              });
          }
        }
      });
    })
    .then(() => res.send({ response: "User Deleted" }))
    .catch((error) => {
      console.log(error);
      res.status(400).json({ error: error });
    });
};

/*************************************************************/
/* ---------- FIXING DATAS --------------------------------- */
/*********************************************************** */
/* Sequelize - Fixing Datas */
const seqFix = async (req, res) => {
  /*
  User.update(
    { gender: 1 },
    {
      where: {
        gender: " male",
      },
    }
  )
    .then(() =>
      User.update(
        { gender: 2 },
        {
          where: {
            gender: "female",
          },
        }
      )
    )
    .then(() =>
      User.update(
        { gender: 3 },
        {
          where: {
            gender: "non binary",
          },
        }
      )
    )
    .then(() =>
      User.update(
        { gender: 3 },
        {
          where: {
            gender: "nonbinary",
          },
        }
      )
    )
    .then(() =>
      User.update(
        { gender: 1 },
        {
          where: {
            gender: "male",
          },
        }
      )
    )
    .then((resp) => res.status(200).json({ res: resp }));
*/

  try {
    // Fetch all file names from the database
    const dbImages = await Image.findAll({ attributes: ["image"] });
    const dbFileNames = dbImages.map((image) => image.image);

    // Read all file names from the directory
    fs.readdir(directoryPathTalents, (err, files) => {
      if (err) {
        console.error(`Error reading directory: ${err}`);
      } else {
        // Iterate through files in the directory
        files.forEach((file) => {
          // Check if the file is not present in the database
          if (!dbFileNames.includes(file)) {
            // If not present in the database, delete the file
            deleteFile(`${directoryPathTalents}${file}`);
          }
        });
      }
    });
  } catch (error) {
    console.error("Error fetching images from database:", error);
  }
  /*
  try {
    // Read all file names from the directory
    fs.readdir(directoryPathTalents, (err, files) => {
      if (err) {
        console.error(`Error reading directory: ${err}`);
      } else {
        // Fetch all image filenames from the database
        Image.findAll({ attributes: ["image"] })
          .then((dbImages) => {
            const dbFileNames = dbImages.map((image) => image.image);

            // Iterate through entries in the database
            dbFileNames.forEach((image) => {
              // Check if the file is not present in the directory
              if (!files.includes(image)) {
                // If not present in the directory, remove the entry from the database
                removeFromDatabase(image);
              }
            });
          })
          .catch((error) => {
            console.error("Error fetching images from database:", error);
          });
      }
    });
  } catch (error) {
    console.error("Error removing orphaned entries:", error);
  }*/
};

/*************************************************************/
/* ---------- FEATURE REMOVE ------------------------------- */
/*********************************************************** */
/* Sequelize - Feature Remove */
const seqFeatRemove = async (req, res) => {
  /* Delete User/Features */
  try {
    Feature.destroy({
      where: {
        ...req.body,
      },
    });
    res.send({ response: "Feature Removed" });
  } catch (error) {
    res.status(400).json({ response: error });
  }
};

/*************************************************************/
/* ---------- PROFILE EDIT --------------------------------- */
/* ---------- MOD-04-APR-2024 ------------------------------ */
/*********************************************************** */
/* Sequelize - Set Profile Image */
const seqProfileImg = async (req, res) => {
  const id = req.body.id;
  const fileName = req.body.fileName;

  try {
    const user = await User.findOne({ where: { id: id } });

    if (!user) {
      console.log("User not found");
      return res.status(404).json({ message: "User not found" });
    }

    const imageName = user.profileimage;

    // Delete the existing image file
    fs.unlink(`${directoryPathProfile}${imageName}`, (err) => {
      if (err) {
        console.error("Error deleting image:", err);
        // Handle error - return an appropriate response or throw an error
      } else {
        console.log("Previous Profile Image deleted successfully !");
      }
    });

    // Resize and save new image
    await sharp(`${directoryPathTalents}${fileName}`)
      .resize({
        height: 400,
        withoutEnlargement: true,
        kernel: sharp.kernel.nearest,
        quality: 100,
      })
      .toFile(path.join(directoryPathProfile, fileName));

    // Update user's profile image
    await User.update({ profileimage: fileName }, { where: { id: id } });

    return res.status(200).json({ message: "Profile Image Updated" });
  } catch (error) {
    console.error("Error updating profile image:", error);
    return res.status(500).json({ message: "Internal server error" });
  }
};

/*************************************************************/
/* ---------- PROFILE IMAGE REMOVE-------------------------- */
/*********************************************************** */
/* Sequelize - Remove Profile Image */
const seqProfileRemove = async (req, res) => {
  /* Set Variables *
  const pid = req.params.id;

  const file = await User.findAll({
    where: {
      id: pid,
    },
  });

  const filterResult = Object.values(file).map((val) => {
    return val;
  });
  const profileFileName = Object.values(filterResult).map((test) => {
    return test.profileimage;
  });

  return User.update(
    { profileimage: null },
    {
      where: {
        id: pid,
      },
    }
  )
    .then(async () => {
      try {
        await rename(
          path.join(`../api/public/images/profiles/`) + profileFileName,
          path.join(`../api/public/images/talents/`) + profileFileName
        );
      } catch (err) {
        if (err) throw err;
        console.log("source.txt was copied to destination.txt");
      }
    })
    .then(() => {
      res.status(200).json({ message: "Profile Image Deleted" });
    })
    .catch((error) => {
      res.status(400).json({ error: error });
    });*/
};

/*************************************************************/
/* ---------- GALLERY IMAGE DELETE FOREVER ----------------- */
/*********************************************************** */
const seqRemoveImg = async (req, res) => {
  /* Set Variables */
  const data = { ...req.body };

  /* Function to delete file with retry */
  const deleteFileWithRetry = async (
    filePath,
    maxRetries = 5,
    retryDelay = 1000
  ) => {
    let retries = 0;
    while (retries < maxRetries) {
      try {
        // Attempt to unlink the file
        await fs.promises.unlink(filePath);
        console.log(`File ${filePath} deleted successfully.`);
        return true; // Exit function if deletion succeeds
      } catch (error) {
        if (error.code === "EBUSY") {
          console.log(
            `File ${filePath} is busy. Retrying after ${retryDelay}ms...`
          );
          await new Promise((resolve) => setTimeout(resolve, retryDelay)); // Wait for retryDelay milliseconds
          retries++;
        } else {
          console.error(`Error deleting file ${filePath}:`, error);
          return false; // Exit function if deletion fails due to other reasons
        }
      }
    }
    console.error(
      `Unable to delete file ${filePath} after ${maxRetries} retries.`
    );
    return false; // Exit function if max retries exceeded
  };

  /* Delete Images from DB by gid and userId */
  try {
    // Destroy the image record from the database
    await Image.destroy({ where: data });

    // Delete the file from the directory with retry
    const filePath = `${directoryPathTalents}${req.body.image}`;
    const deletionResult = await deleteFileWithRetry(filePath);

    if (deletionResult) {
      res.status(200).json({ response: "Image Deleted !" });
    } else {
      res.status(400).json({ response: "Error deleting image" });
    }
  } catch (error) {
    // Handle any errors that occur during the delete operation
    console.error("Error deleting image:", error);
    res.status(400).json({ response: "Error deleting image" });
    // Respond with an error if needed
  }
};

/*************************************************************/
/* ---------- CREATE SHORTLIST --------------- */
/*********************************************************** */
/* Sequelize - Create ShortList */
const seqShortList = async (req, res) => {
  let profileData = { ...req.body };
  try {
    return await ShortList.findOne({
      where: { shortlist_name: profileData.name },
    }).then((data) => {
      if (data) {
        res.status(201).json({ message: "Already Exist !", data: data });
      } else {
        try {
          return ShortList.create({
            shortlist_name: profileData.name,
            shortlist_data: profileData.data,
          }).then(() => {
            res
              .status(200)
              .json({ message: `Shortlist "${profileData.name}" Created` });
          });
        } catch (error) {
          res.status(400).json({ message: "Not Successfully !" });
        }
      }
    });
  } catch (err) {
    console.log(err);
  }
};

/*************************************************************/
/* ---------- REMOVE SHORTLIST ----------------------------- */
/*********************************************************** */
/* Sequelize - ShortList Remove */
const seqRemoveProfile = async (req, res) => {
  const data = { ...req.body };
  try {
    return ShortList.destroy({
      where: {
        id: data.id,
      },
    }).then(() =>
      ShortList.findAll().then((data) => {
        //    console.log(JSON.stringify(data));
        res.status(200).json({ message: "ShortList Removed", data: data });
      })
    );
  } catch (error) {
    res.status(400).json({ message: "Not Successfully !" });
  }
};

/*************************************************************/
/* ---------- MANAGE SHORTLIST UPDATE ---------------------- */
/*********************************************************** */
/* Sequelize - Select User */
const seqSetProfile = async (req, res) => {
  const data = { ...req.body };

  try {
    if (data.id && !data.name) {
      await ShortList.update(
        {
          shortlist_data: data.data,
        },
        {
          where: {
            id: data.id,
          },
        }
      ).then(() =>
        ShortList.findAll().then((data) =>
          res
            .status(200)
            .json({ message: "Profile Removed Successfully", data: data })
        )
      );
    } else if (data.id && data.name) {
      await ShortList.update(
        {
          shortlist_name: data.name,
        },
        {
          where: {
            id: data.id,
          },
        }
      ).then(() =>
        ShortList.findAll().then((data) =>
          res
            .status(200)
            .json({ message: "Profile Removed Successfully", data: data })
        )
      );
    } else if (data.id === 0 && data.name) {
      ShortList.findOne({ where: { shortlist_name: data.name } }).then(
        (getData) => {
          const oldData = JSON.parse(getData.shortlist_data);
          let newData = JSON.parse(data.data);
          newData = [...new Set(oldData.concat(newData))];
          newData = JSON.stringify(newData);
          try {
            return ShortList.update(
              {
                shortlist_data: newData,
              },
              {
                where: {
                  shortlist_name: data.name,
                },
              }
            ).then(() =>
              res.status(200).json({
                message: `Profile ${data.name}Updated Successfully`,
              })
            );
          } catch (error) {
            res.status(400).json({ message: "Error !" });
          }
        }
      );
    }
  } catch (error) {
    res.status(400).json({ message: "UnSuccesfuly" });
  }
};

/*************************************************************/
/* ---------- MANAGE SHARED SHORTLIST UPDATE --------------- */
/*********************************************************** */
/* Sequelize - Select User */
const seqUpdateSharedShortList = async (req, res) => {
  const data = { ...req.body };

  if (data.notes && !data.clientId) {
    try {
      const response = await ClientShare.update(
        {
          client_notes: data.notes,
        },
        {
          where: {
            id: data.id,
          },
        }
      ).then(() =>
        ClientShare.findAll({
          where: {
            id: data.id,
          },
        }).then((data) => {
          /// console.log(data);
          res
            .status(200)
            .json({ message: "Profile Removed Successfully", data: data });
        })
      );
    } catch (error) {
      res.status(400).json({ message: "UnSuccesfuly" });
    }
  } else if (data.clientId) {
    let shared = await ClientShare.findAll({ where: { id: data.clientId } });
    shared = JSON.parse(shared[0].dataValues.client_data);

    const updatedData = shared.map((item) => {
      if (item.id === Number(data.id)) {
        const newData = data.data.length === 0 ? "[]" : data.data;
        return {
          ...item,
          shortlist_data: newData, // Update shortlist_data with the new data
        };
      }
      return item;
    });

    try {
      const updateData = {
        client_data: JSON.stringify(updatedData),
      };

      // Conditionally add the notes field if it exists
      if (data.notes !== undefined && data.notes !== null) {
        updateData.notes = data.notes;
      }

      if (data.clientId) {
        const response = await ClientShare.update(updateData, {
          where: {
            id: data.clientId,
          },
        }).then(() =>
          ClientShare.findAll({
            where: {
              id: data.clientId,
            },
          }).then((data) => {
            /// console.log(data);
            res
              .status(200)
              .json({ message: "Profile Removed Successfully", data: data });
          })
        );
      }
    } catch (error) {
      res.status(400).json({ message: "UnSuccesfuly" });
    }
  }
};

/*************************************************************/
/* ---------- OVERWRITE SHORTLIST -------------------------- */
/*********************************************************** */
/* Sequelize - Create User */
const seqCreateProfile = async (req, res) => {};

/*************************************************************/
/* ---------- FIND PROFILES -------------------------------- */
/*********************************************************** */
const seqGetProfile = async (req, res) => {
  try {
    return await Profiles.findAll({
      group: ["profilename"],
      //attributes: ["profilename"],
    }).then((response) => res.send(response));
  } catch (error) {
    console.log(error);
  }
};

/*************************************************************/
/* ---------- ADD NEW USER --------------------------------- */
/*********************************************************** */
/* Sequelize - Create User */
const seqCreate = async (req, res) => {
  /* Set Variables */
  const uData = { ...req.body };
  let dataId = "";
  let image = "";
  let profileImage = "";
  let galleryImages = "";

  /* REV 0.2
    Provide Profile Image FileName */
  if (req.files.uploadImage && req.files.uploadImage.length > 0) {
    const proccesedImage = await processImage(
      req.files.uploadImage,
      //directoryPathProfile
      directoryPathTalents
    );
    profileImage = proccesedImage.filename.replace(/\..+$/, "") + `.webp`;
    sharp(directoryPathTalents + profileImage)
      .resize({ height: 300, withoutEnlargement: true })
      .toFile(path.join(directoryPathProfile, profileImage));
  }

  /* Encrypt Password */
  let password = req.body.password;
  /*{
    !password && password == "111111";
  }*/
  const salt = genSaltSync(10);
  password = hashSync(password, salt);

  /* Create User */
  return User.create({
    username: uData.username,
    email: uData.email,
    password: password,
    role: uData.user,
    prefix: uData.prefix,
    phone: uData.phone,
    firstname: uData.firstname,
    lastname: uData.lastname,
    birthyear: uData.birthyear,
    age: uData.age,
    height: uData.height,
    gender: uData.gender,
    shoe: uData.shoe,
    clothes: uData.clothes,
    active: uData.active,
    startingdate: uData.startingdate,
    endingdate: uData.endingdate,
    exclusive: uData.exclusive,
    usertext: uData.usertext,
    profileimage: profileImage,
  })
    .then((userData) => {
      /* Create User Features*/
      dataId = userData.id;
      let feat = [];
      let extraFeat = [];
      feat = [JSON.parse(uData.feat)];
      extraFeat = JSON.parse(uData.extrafeat).map((res) => res);
      extraFeat.map((data) => {
        Feature.create({
          fid: userData.id,
          ftype: 4,
          id: data.id,
          name: "extra",
          feat: data.feat,
          userId: userData.id, // Associate with userID
        });
      });
      feat.map((data) => {
        data.eyecolor &&
          Feature.create({
            fid: userData.id,
            ftype: 1,
            name: "eyecolor",
            feat: data.eyecolor,
            userId: userData.id, // Associate with userID
          });
        data.haircolor &&
          Feature.create({
            fid: userData.id,
            ftype: 2,
            name: "haircolor",
            feat: data.haircolor,
            userId: userData.id, // Associate with userID
          });
        data.ethnicity &&
          Feature.create({
            fid: userData.id,
            ftype: 3,
            name: "ethnicity",
            feat: data.ethnicity,
            userId: userData.id, // Associate with userID
          });
        data.nationality &&
          Feature.create({
            fid: userData.id,
            ftype: 5,
            name: "nationality",
            feat: data.nationality,
            userId: userData.id, // Associate with userID
          });
      });
    })
    .then(async () => {
      /* REV
    Create / Update User Profile Image */
      profileImage &&
        Image.create({
          gid: dataId,
          image: profileImage,
          userId: dataId, // Associate with userID
        }).catch((err) => {
          console.error(err);
          throw err;
        });

      /* REV 0.2 
    Provide Gallery Images FileNames */
      if (req.files.uploadImages) {
        galleryImages = await processImages(
          req.files.uploadImages,
          directoryPathTalents
        );
      }
      /* REV
    Create / Add Images into Galleries */
      galleryImages &&
        galleryImages.forEach((image) => {
          Image.create({
            gid: dataId,
            image: image.filename.replace(/\..+$/, "") + `.webp`,
            userId: dataId, // Associate with userID
          }).catch((err) => {
            console.error(err);
            throw err;
          });
        });
    })
    .catch((error) => {
      {
        error && console.log(error);
      }
      res.status(409).json({ error: error });
    })
    .then((response) => {
      res.send(response);
    });
};

/********************************************************************
 *  UPDATE USER - REV 0.2 - AUGUST 2024                            *
 *  -----------------------------------------------------------------
 *  Description: Perform user update based on specified conditions. *
 *  Date: August 01, 2024                                          *
 * *****************************************************************/
const seqUpdate = async (req, res) => {
  const id = req.params.id;
  const uData = { ...req.body };
  let profileImage = "";
  let galleryImages = "";

  try {
    let featData = JSON.parse(uData.feat);
    let fullFeatData = [
      featData.eyecolor && {
        name: "eyecolor",
        feat: featData.eyecolor,
        ftype: 1,
        userId: id,
        fid: id,
      },
      featData.haircolor && {
        name: "haircolor",
        feat: featData.haircolor,
        ftype: 2,
        userId: id,
        fid: id,
      },
      featData.ethnicity && {
        name: "ethnicity",
        feat: featData.ethnicity,
        ftype: 3,
        userId: id,
        fid: id,
      },
      featData.nationality && {
        name: "nationality",
        feat: featData.nationality,
        ftype: 5,
        userId: id,
        fid: id,
      },
    ].filter(Boolean);

    // Handle each feature
    for (const feature of fullFeatData) {
      //  console.log(`Processing feature: ${feature.name}`);

      // Find existing records
      const existingFeatures = await Feature.findAll({
        where: {
          name: feature.name,
          [Op.or]: [{ fid: feature.fid }, { userId: feature.userId }],
        },
      });

      if (existingFeatures.length > 1) {
        // Delete all existing records if duplicates are found
        await Feature.destroy({
          where: {
            name: feature.name,
            [Op.or]: [{ fid: feature.fid }, { userId: feature.userId }],
          },
        });
        console.log(
          `Multiple same feature entries detected. All deleted: ${feature.name}`
        );
      }

      // Update or create the feature
      if (existingFeatures.length === 1) {
        // If exactly one record exists, update it
        await Feature.update(
          {
            ftype: feature.ftype,
            feat: feature.feat,
          },
          {
            where: {
              name: feature.name,
              [Op.or]: [{ fid: feature.fid }, { userId: feature.userId }],
            },
          }
        );
        //   console.log(`Feature updated: ${feature.name}`);
      } else {
        // If no record exists (or if we deleted existing records), create a new one
        await Feature.create(feature);
        //   console.log(`New feature created: ${feature.name}`);
      }
    }

    /* REV 0.2
    Provide Profile Image FileName */
    if (req.files.uploadImage && req.files.uploadImage.length > 0) {
      const proccesedImage = await processImage(
        req.files.uploadImage,
        //directoryPathProfile
        directoryPathTalents
      );
      profileImage = proccesedImage.filename.replace(/\..+$/, "") + `.webp`;
      sharp(directoryPathTalents + profileImage)
        .resize({ height: 300, withoutEnlargement: true })
        .toFile(path.join(directoryPathProfile, profileImage));

      await Image.create({
        gid: id,
        image: profileImage,
        userId: id,
      });
    }

    let extraFeat = JSON.parse(uData.extrafeat);
    console.log(extraFeat);
    await Promise.all(
      extraFeat.map(async (extrafeature) => {
        const exist = await Feature.findOne({
          where: { feat: extrafeature.feat, userId: id },
        });
        if (!exist) {
          await Feature.upsert(
            {
              fid: id,
              ftype: 4,
              id: extrafeature.id,
              name: "extra",
              feat: extrafeature.feat,
              userId: id,
            },
            {
              where: {
                name: "extra",
                userId: id,
              },
            }
          );
        }
      })
    );

    if (req.files.uploadImages) {
      galleryImages = await processImages(
        req.files.uploadImages,
        directoryPathTalents
      );

      await Promise.all(
        galleryImages.map(async (image) => {
          await Image.create({
            gid: id,
            image: image.filename.replace(/\..+$/, "") + `.webp`,
            userId: id,
          });
        })
      );
    }

    /* Encrypt Password if provided */
    let password = uData.password;
    if (password) {
      const salt = genSaltSync(10);
      password = hashSync(password, salt);
      uData.password = password;
    }
    if (profileImage) {
      uData.profileimage = profileImage;
    }
    await User.update(uData, { where: { id: id } });

    // seqFix();
    res.status(200).json({ response: `UID:${id} - Updated` });
  } catch (error) {
    console.error(error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

/*************************************************************/
/* ---------- ADD EXTRA FEATURES --------------------------- */
/*********************************************************** */
const seqAddExtraFeat = async (req, res) => {
  const fData = { ...req.body };
  console.log(fData);
  try {
    await ExtraFeats.findOne({
      where: {
        feat: fData.feat,
      },
    }).then((e) => {
      if (!e) {
        ExtraFeats.upsert(fData).then((response) => {
          res.status(200).json({ response: `Extra Feat Added` });
        });
      } else {
        res.status(400).json({ response: "Feat already exist !" });
      }
    });
  } catch (error) {
    res.status(400).json({ Error: error });
  }
};

/*************************************************************/
/* ---------- REMOVE EXTRA FEATURES ------------------------ */
/*********************************************************** */
const seqRemoveExtraFeat = async (req, res) => {
  const fData = req.body;
  console.log(fData);
  ExtraFeats.destroy({ where: { id: fData.eid, feat: fData.newfeat } })
    .then(() => res.send({ response: "Extra Feat Deleted" }))
    .catch((error) => {
      res.status(400).json({ error: error });
      console.log(error.message);
    });
};

/*************************************************************/
/* ---------- ADD NEW NATIONALITY -------------------------- */
/*********************************************************** */
const seqAddNewNat = async (req, res) => {
  const fData = req.body.data;
  try {
    await Nationality.findOne({
      where: {
        nationality: fData.nationality,
      },
    }).then((e) => {
      if (!e) {
        Nationality.upsert(fData).then(() => {
          res.status(200).json({ response: `Nationality Added` });
        });
      } else {
        res.status(400).json({ response: "Nationality already exist !" });
      }
    });
  } catch (error) {
    res.status(400).json({ Error: error });
  }
};

/*************************************************************/
/* ---------- ADD NEW NATIONALITY -------------------------- */
/*********************************************************** */
const seqRemoveNat = async (req, res) => {
  const deleteData = { ...req.body };
  // console.log(deleteData);
  if (deleteData) {
    try {
      await Nationality.destroy({ where: deleteData }).then(() => {
        res.status(200).json({
          response: `Nationality : ${deleteData.nationality} - DELETED`,
        });
      });
    } catch (error) {
      res.status(400).json({ response: "Nationality already exist !" });
    }
  }
};

/*************************************************************/
/* ---------- SET/DELETE/UPDATE KEY SEARCH FILTERS --------- */
/*********************************************************** */
const seqFilters = async (req, res) => {
  const filterData = { ...req.body };
  try {
    if (!filterData.filter_name) {
      Filters.findOne({
        where: {
          filter_uid: filterData.filter_uid,
        },
      }).then((existID) => {
        if (!existID) {
          Filters.create(filterData).then((response) =>
            res.status(200).json({ response: response })
          );
        } else {
          Filters.update(filterData, {
            where: {
              filter_uid: filterData.filter_uid,
            },
          });
        }
      });
    } else if (filterData.filter_name) {
      Filters.update(
        { filter_name: filterData.filter_name },
        {
          where: {
            filter_uid: filterData.filter_uid,
          },
        }
      );
    } else if (!filterData.filter_name && filterData.filter_data) {
      Filters.update(
        { filter_data: filterData.filter_data },
        {
          where: {
            filter_uid: filterData.filter_uid,
          },
        }
      );
    }
    if (filterData.delete_data === "yes") {
      Filters.destroy({
        where: {
          filter_uid: filterData.filter_uid,
        },
        truncate: true,
        restartIdentity: true,
      });
    }
  } catch (error) {
    console.log(error);
    res.status(400).json({ error: error });
  }
};

/*************************************************************/
/* ---------- SET EXTRA FILTERS ---------------------------- */
/*********************************************************** */
const seqExtraFilters = async (req, res) => {
  const filterData = { ...req.body };
  // console.log(filterData);

  try {
    ExtraFilters.create(filterData).then((data) => {
      res.status(200).json(data);
    });
  } catch (error) {
    console.log(error);
    res.status(400).json({ error: error });
  }
};

/*************************************************************/
/* ---------- DELETE EXTRA FILTERS ------------------------- */
/*********************************************************** */
const seqDeleteFilters = async (req, res) => {
  const filterData = { ...req.body };
  console.log(filterData);

  try {
    ExtraFilters.destroy({
      where: filterData,
    }).then((data) => {
      res.status(200).json(data);
    });
  } catch (error) {
    console.log(error);
    res.status(400).json({ error: error });
  }
};

/*************************************************************/
/* ---------- ADD PROJET ----------------------------------- */
/*********************************************************** */
const seqProjectAdd = async (req, res) => {
  const data = req.body;
  data.project_date = data.project_date ? data.project_date : new Date();
  data.project_update = data.project_update ? data.project_update : new Date();

  const totalAmount = parseInt(data.project_cost) || 0;
  const totalPayed = parseInt(data.project_payed) || 0;

  if (totalPayed === 0 && totalAmount === 0) {
    data.project_state = 0;
  } else if (totalAmount === totalPayed) {
    data.project_state = 1;
  } else if (totalAmount > totalPayed && totalPayed !== 0) {
    data.project_state = 2;
  } else if (
    (totalAmount < totalPayed && totalPayed !== 0) ||
    (totalAmount === 0 && totalPayed > 0)
  ) {
    data.project_state = 3;
  }

  if (data) {
    try {
      // Create the project entry*/
      await Projects.create(data); /*.then((datas) => {
        console.log(datas);
      });
*/
      res.status(200).json({ create: true });
    } catch (error) {
      console.log(error);
      res.status(500).json({ error: error.message });
    }
  } else {
    res.status(400).json({ error: "Missing required fields" });
  }
};

/*************************************************************/
/* ---------- PROJECT UPDATE ------------------------------- */
/*********************************************************** */
const seqProjectUpdate = async (req, res) => {
  const id = req.params.id;
  const data = req.body;

  //console.log(data, id);
  // data.project_date = data.project_date ? data.project_date : new Date();

  const totalAmount = parseInt(data.project_cost) || 0;
  const totalPayed = parseInt(data.project_payed) || 0;

  if (!data.project_state) {
    if (
      (totalPayed === 0 && totalAmount === 0) ||
      (totalAmount > totalPayed && totalPayed === 0)
    ) {
      data.project_state = 0;
    } else if (totalAmount === totalPayed) {
      data.project_state = 1;
    } else if (totalAmount > totalPayed && totalPayed !== 0) {
      data.project_state = 2;
    } else if (
      (totalAmount < totalPayed && totalPayed !== 0) ||
      (totalAmount === 0 && totalPayed > 0)
    ) {
      data.project_state = 3;
    }
  }

  if (
    req.body.project_state !== 0 &&
    req.body.project_update !== new Date().toISOString().substring(0, 10)
  ) {
    data.project_update = req.body.project_update;
  }

  if (req.body.project_date !== new Date().toISOString().substring(0, 10)) {
    data.project_date = req.body.project_date;
  } else {
    // data.project_date = req.body.project_date;
  }

  if (data.project_state === 1) {
    data.project_payed = data.project_cost;
  } else {
    data.project_payed = data.project_payed ? data.project_payed : 0;
  }

  if (data) {
    try {
      await Projects.update(data, { where: { project_id: id } });

      res.status(200).json({ update: true });
    } catch (error) {
      console.log(error);
      res.status(500).json({ error: error.message });
    }
  } else {
    res.status(400).json({ error: "Missing required fields" });
  }
};

/*************************************************************/
/* ----------DELETE PROJET ----------------------------------- */
/*********************************************************** */
const seqProjectDelete = async (req, res) => {
  const { id } = req.body;

  if (id) {
    try {
      await Projects.destroy({ where: { project_id: id } });
      res.status(200).json({ deleted: true });
    } catch (error) {
      console.log(error);
      res.status(500).json({ error: error.message });
    }
  } else {
    res.status(400).json({ error: "Missing required field: id" });
  }
};

/*************************************************************/
/* ---------- ADD PROJET TALENT ---------------------------- */
/*********************************************************** */
const seqTalentAdd = async (req, res) => {
  const id = req.params.id;
  const data = req.body;
  const firstName = data.firstname ? data.firstname : "";
  const lastName = data.lastname ? data.lastname : "";
  const address = data.address ? data.address : "";
  const idCard = data.id_card ? data.id_card : "";
  let state = 0;
  let createdUserId;

  //console.log(data.talent_total_payed, data.talent_total_amount, state);

  if (firstName || lastName || address || idCard) {
    try {
      const newUser = await User.create({
        firstname: firstName,
        lastname: lastName,
        address: address,
        id_card: idCard,
        active: 0,
      });
      createdUserId = newUser.id;
    } catch (error) {
      console.log(error);
      return res.status(500).json({ error: "Error creating user" });
    }
  }

  const totalAmount = parseInt(data.talent_total_amount) || 0;
  const totalPayed = parseInt(data.talent_total_payed) || 0;

  if (totalPayed === 0 && totalAmount === 0) {
    state = 0;
  } else if (totalAmount === totalPayed) {
    state = 1;
  } else if (totalAmount > totalPayed && totalPayed !== 0) {
    state = 2;
  } else if (
    (totalAmount < totalPayed && totalPayed !== 0) ||
    (totalAmount === 0 && totalPayed > 0)
  ) {
    state = 3;
  }

  if (data) {
    try {
      // const projectId = data.talent_project_id;
      const userId = data.user_id ?? createdUserId;

      await ProjectsTalents.create({
        talent_project_id: id,
        user_id: userId,
        talent_notes: data.talent_notes,
        talent_date: new Date(),
        talent_update: new Date(),
        talent_total_amount: data.talent_total_amount,
        talent_total_payed: data.talent_total_payed,
        talent_state: state,
        currency: data.currency,
      });

      res.status(200).json({ update: true });
    } catch (error) {
      console.log(error);
      res.status(500).json({ error: error.message });
    }
  } else {
    res.status(400).json({ error: "Missing required fields" });
  }
};

/*************************************************************/
/* ----------  PROJECT TALENT UPDATE ----------------------- */
/*********************************************************** */
const seqTalentUpdate = async (req, res) => {
  const id = req.params.id;
  const userId = req.params.userId;
  const data = req.body;

  //data.talent_date = data.talent_date ? data.talent_date : new Date();

  const totalAmount = parseInt(data.talent_total_amount) || 0;
  const totalPayed = parseInt(data.talent_total_payed) || 0;

  if (!data.talent_state) {
    if (
      (totalPayed === 0 && totalAmount === 0) ||
      (totalAmount > totalPayed && totalPayed === 0)
    ) {
      data.talent_state = 0;
    } else if (totalAmount === totalPayed) {
      data.talent_state = 1;
    } else if (totalAmount > totalPayed && totalPayed !== 0) {
      data.talent_state = 2;
    } else if (
      (totalAmount < totalPayed && totalPayed !== 0) ||
      (totalAmount === 0 && totalPayed > 0)
    ) {
      data.talent_state = 3;
    }
  }

  if (
    req.body.talent_state !== 0 &&
    req.body.talent_update !== new Date().toISOString().substring(0, 10)
  ) {
    data.talent_update = req.body.talent_update;
    console.log("update changed");
  }

  if (req.body.talent_date !== new Date().toISOString().substring(0, 10)) {
    data.talent_date = req.body.talent_date;
  }

  if (data.talent_state === 1) {
    data.talent_total_payed = data.talent_total_amount;
  } else {
    data.talent_total_payed = data.talent_total_payed
      ? data.talent_total_payed
      : 0;
  }

  try {
    if (userId) {
      await User.update(data, {
        where: { id: userId },
      });
    }

    await ProjectsTalents.update(data, {
      where: { talent_id: id },
    });

    res.status(200).json({ update: true });
  } catch (error) {
    console.log(error);
    res.status(500).json({ error: error.message });
  }
};

/*************************************************************/
/* ---------- DELETE PROJET TALENT ------------------------- */
/*********************************************************** */
const seqTalentDelete = async (req, res) => {
  const { id } = req.body;

  if (id) {
    try {
      await ProjectsTalents.destroy({ where: { talent_id: id } });
      res.status(200).json({ deleted: true });
    } catch (error) {
      console.log(error);
      res.status(500).json({ error: error.message });
    }
  } else {
    res.status(400).json({ error: "Missing required field: id" });
  }
};

/*************************************************************/
/* ---------- MANAGE SHARED ROUTES TO CLIENT --------------- */
/*********************************************************** */
const clientSharedRoutes = async (req, res) => {
  const data = req.body;

  try {
    if (data.option === "remove") {
      //  console.log("shared link and route deleted");

      const response = await ClientsRoutes.findAll({
        where: {
          path: data.token,
        },
      });

      if (response.length > 0) {
        await ClientsRoutes.destroy({
          where: {
            path: data.token,
          },
        });
      }

      await ClientShare.destroy({
        where: {
          client_token: data.token,
        },
      });

      return res
        .status(200)
        .json({ deleted: true, message: "Shared link and route deleted" });
    }

    if (data.option && data.option !== false) {
      //  console.log("shared route enabled");

      await ClientShare.update(
        { client_linked: 1 },
        { where: { client_token: data.token } }
      );

      await ClientsRoutes.create({
        sid: data.id,
        cid: data.cid,
        path: data.token,
      });
      return res
        .status(200)
        .json({ linked: true, message: "Shared Route Added" });
    }

    if (!data.option && data.option !== true) {
      //   console.log("shared route disabled");

      await ClientShare.update(
        { client_linked: 0 },
        { where: { client_token: data.token } }
      );

      await ClientsRoutes.destroy({
        where: {
          sid: data.id,
          path: data.token,
        },
      });
      return res
        .status(200)
        .json({ linked: false, message: "Shared Route removed" });
    }
  } catch (error) {
    return res.status(500).json({ error: error.message });
  }
};

/*************************************************************/
/* ---------- SHARE TO CLIENT ------------------------------ */
/*********************************************************** */
const sharedClient = async (req, res) => {
  const data = req.body;
  const token = crypto.randomBytes(50).toString("hex");
  //+ uuidv4();
  const expiryTime = Date.now() + 24 * 60 * 60 * 1000; // 24 hours expiry
  // console.log(token, data);

  try {
    const shareResponse = await ClientShare.create({
      ...data,
      client_token: token,
    });

    // Accessing the new ID
    const newId = shareResponse.id || shareResponse.dataValues?.id; // Safely access id
    const routesResponse = await ClientsRoutes.create({
      cid: data.client_id,
      sid: newId,
      path: token,
    });
    res.status(200).json({ create: true, sharedLink: token });
  } catch (error) {
    console.log(error);
    res.status(500).json({ error: error.message });
  }
};

/*************************************************************/
/* ---------- UPDATE SHARE TO CLIENT ----------------------- */
/*********************************************************** */
const updateShared = async (req, res) => {
  const id = req.params.id;
  const data = req.body;

  try {
    const shareResponse = await ClientShare.update(data, { where: { id: id } });
    res.status(200).json({
      create: true,
      shared: shareResponse,
    });
  } catch (error) {
    console.log(error);
    res.status(500).json({ error: error.message });
  }
};

/*************************************************************/
/* ---------- HANDLE UPLOAD IMAGE -------------------------- */
/*********************************************************** */
const handleImage = async (uploadImage) => {
  let profileImage = "";

  if (uploadImage && uploadImage.length > 0) {
    const proccesedImage = await processImage(
      uploadImage,
      directoryPathTalents
    );
    profileImage = proccesedImage.filename.replace(/\..+$/, "") + `.webp`;

    await sharp(directoryPathTalents + profileImage)
      .resize({ height: 300, withoutEnlargement: true })
      .toFile(path.join(directoryPathClients, profileImage));
  }

  return profileImage;
};

/*************************************************************/
/* ---------- ADD CLIENT ----------------------------------- */
/*********************************************************** */
const addClient = async (req, res) => {
  const data = JSON.parse(req.body.formData);

  try {
    const profileImage = await handleImage(req.files.uploadImage);

    const uData = {
      ...data,
      ...(profileImage && { client_profileimage: profileImage }),
    };
    ClientsModel.create(uData);
    res.status(200).json({ created: true, message: "Client Created" });
  } catch (error) {
    console.log(error);
    res.status(500).json({ error: error.message });
  }
};

/*************************************************************/
/* ---------- UPDATE CLIENT -------------------------------- */
/*********************************************************** */
const updateClient = async (req, res) => {
  const data = JSON.parse(req.body.formData);
  const id = JSON.parse(req.params.userId);

  try {
    const profileImage = await handleImage(req.files.uploadImage);

    const uData = {
      ...data,
      ...(profileImage && { client_profileimage: profileImage }),
    };

    ClientsModel.update(uData, { where: { id: id } });
    res.status(200).json({ created: true, message: "Client Updated" });
  } catch (error) {
    console.log(error);
    res.status(500).json({ error: error.message });
  }
};

/*************************************************************/
/* ---------- DELETE CLIENT -------------------------------- */
/*********************************************************** */
const deleteClient = async (req, res) => {
  const id = req.params.userId;

  try {
    // Delete client shares first to avoid foreign key constraint issues
    await ClientShare.destroy({ where: { client_id: id } });

    // Then delete the client
    await ClientsModel.destroy({ where: { id: id } });

    res.status(200).json({ deleted: true, message: "Client Deleted" });
  } catch (error) {
    if (error.name === "SequelizeForeignKeyConstraintError") {
      res.status(400).json({
        error: error.message,
        message: "Cannot DELETE, there is a project linked to the client!",
      });
    } else {
      console.log(error);
      res.status(500).json({
        error: error.message,
        message: "An unexpected error occurred",
      });
    }
  }
};

/* Exports Modules*/
module.exports = {
  seqCreate,
  seqRemove,
  seqUpdate,
  seqRemoveImg,
  seqProfileImg,
  seqProfileRemove,
  seqFeatRemove,
  seqAddExtraFeat,
  seqRemoveExtraFeat,
  seqCreateProfile,
  seqRemoveProfile,
  seqSetProfile,
  seqGetProfile,
  seqAddNewNat,
  seqFix,
  seqRemoveNat,
  seqFilters,
  seqExtraFilters,
  seqDeleteFilters,
  seqShortList,
  seqCropper,
  seqProjectUpdate,
  seqProjectAdd,
  seqTalentUpdate,
  seqTalentAdd,
  seqTalentDelete,
  seqProjectDelete,
  sharedClient,
  seqUpdateSharedShortList,
  addClient,
  updateClient,
  deleteClient,
  clientSharedRoutes,
  updateShared,
};
