const db = require("../config/Db.js");
const { compareSync } = require("bcryptjs");
const jwt = require("jsonwebtoken");
const User = require("../models/UsersModel.js");
const Feature = require("../models/FeaturesModel.js");
const Tags = require("../models/TagsModel.js");
const Image = require("../models/ImagesModel.js");
const { Op } = require("sequelize");
const ExtraFeats = require("../models/ExtraFeatsModel.js");
const Profiles = require("../models/ProfilesModel.js");
const Nationality = require("../models/Nationality.js");
const Filters = require("../models/Filters.js");
const ExtraFilters = require("../models/ExtraFilters.js");
const ShortList = require("../models/ShortList.js");
//const Settings = require("../models/Settings.js");
const FeatureType = require("../models/FeatureTypeModel.js");
const ProjectsTalents = require("../models/ProjectsTalentsModel.js");
const Projects = require("../models/ProjectsModel.js");
const Clients = require("../models/ClientsModel.js");
const ClientShare = require("../models/ClientShare.js");
const ClientsRoutes = require("../models/ClientsRoutes.js");
const ShortListIDs = require("../models/ShortListId.js");
const { v4: uuidv4 } = require("uuid");
const CryptoJS = require("crypto-js");
const SECRET_KEY = process.env.SECRET_KEY;

const encryptData = (data) => {
  try {
    const stringifiedData = JSON.stringify(data);
    const encrypted = CryptoJS.AES.encrypt(
      stringifiedData,
      SECRET_KEY
    ).toString();
    return encrypted;
  } catch (error) {
    console.error("Encryption Error:", error);
    return null;
  }
};

/* LOGIN USER */
const LogIn = async (req, res) => {
  const logInData = { ...req.body };
  try {
    return await User.findOne({
      where: {
        email: logInData.email,
        [Op.or]: [{ id: 1 }, { id: 2 }],
      },
    }).then((userCheck) => {
      if (userCheck) {
        const checkPassword = compareSync(
          logInData.password,
          userCheck.password
        );
        if (checkPassword) {
          req.session.userId = userCheck.id;
          req.session.save();

          const userData = {
            id: userCheck.id,
            email: userCheck.email,
            role: userCheck.role,
            firstname: userCheck.firstname,
            lastname: userCheck.lastname,
          };

          const token = jwt.sign(
            { id: userCheck.id, email: userCheck.email },
            process.env.SESS_SECRET,
            { expiresIn: "1d" }
          );

          if (token) {
            res.status(200).json({
              loggedIn: userData,
              session: req.sessionID,
              token: token,
            });
          } else {
            res.status(400).send("Invalid Token!");
          }
        } else {
          res.status(400).send("Invalid Password!");
        }
      } else {
        res.status(400).send("User Not Found or not Admin !");
      }
    });
  } catch (error) {
    res.status(400).send(error);
  }
};

/* LOGOUT USER */
const LogOut = (req, res) => {
  let id = req.params.id;
  if (id === undefined) {
    id = [0];
  } else {
    /* Force Clear DB Sessions by Logout ID */
    query = `DELETE FROM sessions WHERE data LIKE ?`;
    db.query(query, [`%${id}%`], (err, report) => {
      res.status(200).json("Session Cleared");
    });
  }
};

/*************************************************************/
/* ---------- LIST PROJECTS -------------------------------- */
/*********************************************************** */
const seqListProjects = async (req, res) => {
  try {
    // Define associations
    Projects.hasMany(ProjectsTalents, { foreignKey: "talent_project_id" }); // Assuming project_id is the foreign key in ProjectsTalents
    ProjectsTalents.belongsTo(Projects, { foreignKey: "talent_project_id" });
    ProjectsTalents.belongsTo(User, { foreignKey: "user_id" }); // Assuming user_id is the foreign key in ProjectsTalents
    User.hasMany(Image);

    // Fetch data from multiple tables
    const [clients, projects, projectsTalents, users, routes, shared, list] =
      await Promise.all([
        Clients.findAll({
          order: [["id", "DESC"]],
        }),
        Projects.findAll(),
        ProjectsTalents.findAll({
          include: {
            model: User,
            order: [["id", "DESC"]],
            attributes: [
              "id",
              "firstname",
              "lastname",
              "email",
              "address",
              "id_card",
              "parent",
              "profileimage",
              "professional",
            ], // Specify the fields you want to include from the User model
          },
        }),
        User.findAll({
          include: [Image],
          order: [["id", "DESC"]],
          attributes: [
            "id",
            "firstname",
            "lastname",
            "email",
            "address",
            "id_card",
            "profileimage",
            "height",
            "shoe",
            "clothes",
            "age",
            "parent",
            "professional",
          ], // Specify the fields you want to fetch
        }),
        ClientsRoutes.findAll(),
        ClientShare.findAll(),
        ShortList.findAll(),
      ]);

    // Send response with transformed data
    res.status(200).json({
      clients: clients,
      projects: projects,
      projectsTalents: projectsTalents,
      users: users,
      routes: routes,
      shared: shared,
      list: list,
    });
  } catch (error) {
    res.status(400).json({ Error: error.message });
  }
};

/*************************************************************/
/* ----------SHARED LIST PROJECTS -------------------------- */
/*********************************************************** */
const seqSharedList = async (req, res) => {
  const {
    includeSharedProjectUsers = "false",
    includeSharedEncrypted = "false",
  } = req.query;

  try {
    // Initialize data variables
    let sharedProjectUsers, sharedEncrypted;

    if (includeSharedProjectUsers === "true") {
      try {
        // Fetch all records from clientshortlist
        const clientShortlists = await ClientShare.findAll({
          attributes: ["client_data"],
        });

        // Initialize an array to hold all IDs
        const allIds = [];

        // Iterate over each record
        clientShortlists.forEach((record) => {
          try {
            // Parse the client_data JSON string
            const clientDataArray = JSON.parse(record.client_data);

            // Iterate over each item in the clientDataArray
            clientDataArray.forEach((clientData) => {
              // Access the shortlist_data inside client_data
              const shortlistDataString = clientData.shortlist_data;

              if (shortlistDataString) {
                try {
                  // Parse the shortlist_data JSON string
                  const shortlistData = JSON.parse(shortlistDataString);

                  if (Array.isArray(shortlistData)) {
                    // Extract IDs from the shortlist_data array
                    shortlistData.forEach((item) => {
                      if (item.id) {
                        allIds.push(item.id);
                      }
                    });
                  }
                } catch (error) {
                  console.error("Error parsing shortlist_data JSON:", error);
                }
              }
            });
          } catch (error) {
            console.error("Error parsing client_data JSON:", error);
          }
        });

        // Remove duplicates (if needed)
        const uniqueIds = [...new Set(allIds)];

        // Fetch users with the unique IDs
        projectUsers = await User.findAll({
          order: [["id", "DESC"]],
          attributes: [
            "id",
            "firstname",
            "lastname",
            "email",
            "address",
            "id_card",
            "profileimage",
            "height",
            "shoe",
            "clothes",
            "age",
            "parent",
            "professional",
          ],
          where: {
            id: {
              [Op.in]: uniqueIds, // Use uniqueIds here
            },
          },
        });
        sharedUsersEncrypted = encryptData(projectUsers);
      } catch (error) {
        console.error("Error fetching shared project users:", error);
      }
    }

    if (includeSharedEncrypted === "true") {
      shared = await ClientShare.findAll();
      sharedEncrypted = encryptData(shared);
    }

    // Construct response object with the exact names you want
    const response = {
      sharedEncrypted: sharedEncrypted,
      sharedUsersEncrypted: sharedUsersEncrypted,
    };

    // Send response with the constructed data
    res.status(200).json(response);
  } catch (error) {
    res.status(400).json({ Error: error.message });
  }
};

/*************************************************************/
/* ---------- LIST USERS ----------------------------------- */
/*********************************************************** */
const seqListAll = async (req, res) => {
  try {
    // Define associations
    User.hasMany(Feature);
    Feature.belongsTo(User);
    User.hasMany(Image);
    Image.belongsTo(User);
    User.hasOne(Nationality);
    Nationality.belongsTo(User);
    Feature.belongsTo(FeatureType, { foreignKey: "ftype" });
    Projects.hasMany(ProjectsTalents, { foreignKey: "talent_project_id" }); // Assuming project_id is the foreign key in ProjectsTalents
    ProjectsTalents.belongsTo(Projects, { foreignKey: "talent_project_id" });
    ProjectsTalents.belongsTo(User, { foreignKey: "user_id" }); // Assuming user_id is the foreign key in ProjectsTalents

    // Fetch data from multiple tables
    const [
      users,
      extra,
      prof,
      list,
      nat,
      filters,
      extrafilters,
      features,
      featureTypes,
    ] = await Promise.all([
      User.findAll({
        include: [
          {
            model: Feature,
            include: [
              {
                model: FeatureType,
                attributes: ["ftype_color", "ftype_name"],
              },
            ],
          },
          Image,
        ],
        order: [["id", "DESC"]],
      }),
      ExtraFeats.findAll(),
      Profiles.findAll(),
      ShortList.findAll(),
      Nationality.findAll(),
      Filters.findAll(),
      ExtraFilters.findAll(),
      Feature.findAll({ include: [FeatureType], group: ["feat"] }),
      FeatureType.findAll(),
    ]);
    // Send response with transformed data
    res.status(200).json({
      users: users,
      extra: extra,
      prof: prof,
      list: list,
      nat: nat,
      filters: filters,
      extrafilters: extrafilters,
      feat: features,
      ftypes: featureTypes,
    });
  } catch (error) {
    res.status(400).json({ Error: error.message });
  }
};

/*************************************************************/
/* ---------- LIST ALL USERS DATA -------------------------- */
/*********************************************************** */
const seqQueryAllData = async (req, res) => {
  User.hasMany(Feature);
  Feature.belongsTo(User);
  User.hasMany(Image);
  Image.belongsTo(User);
  User.hasOne(Nationality);
  Nationality.belongsTo(User);
  Feature.belongsTo(FeatureType, { foreignKey: "ftype" });
  ProjectsTalents.belongsTo(Projects, { foreignKey: "talent_project_id" });
  ProjectsTalents.belongsTo(User, { foreignKey: "user_id" }); // Assuming user_id is the foreign key in ProjectsTalents

  const {
    isUsers = "false",
    isUserFeatures = "false",
    isExtraFeatures = "false",
    isProfiles = "false",
    isShortlist = "false",
    isNationality = "false",
    isExtraFilters = "false",
    isFeatures = "false",
    isFeatureTypes = "false",
    includeGalleries = "false",
    includeClients = "false",
    includeProjects = "false",
    includeProjectTalents = "false",
    includeProjectUsers = "false",
    includeRoutes = "false",
    includeClientShare = "false",
    includeTotalUsers = "false",
    isUserInProject = "false",
    userId = null,
    ftype = null,
  } = req.query;

  try {
    // Initialize data variables
    let onlyUsersData,
      usersData,
      extraFeatures,
      profiles,
      shortlist,
      nationality,
      extrafilters,
      features,
      ftypes,
      gallery,
      clients,
      projects,
      projectTalents,
      projectUsers,
      routes,
      clientShareData,
      totalUsers,
      usersInProject;

    if (includeTotalUsers === "true") {
      // Fetch total count of users
      totalUsers = await Promise.all([User.count()]);
    }

    // Conditionally fetch data
    if (isUsers === "true") {
      /*   // Fetch all records from ProjectsTalents
      const projectTalents = await ProjectsTalents.findAll({
        attributes: ["user_id"], // Specify the fields you want to retrieve
      });

      // Extract user_id values from the results
      const userIds = projectTalents.map((talent) => talent.user_id);

      // Optionally, remove duplicates if needed
      const uniqueUserIds = [...new Set(userIds)];
*/
      onlyUsersData = await User.findAll({
        order: [["id", "DESC"]],
        where: userId ? { id: userId } : {},
        attributes: {
          exclude: ["createdAt", "updatedAt, password"], // Exclude these fields from the result
        },
      });
    }

    if (isUserInProject === "true") {
      usersInProject = await User.findAll({
        order: [["id", "DESC"]],
        where: userId ? { id: userId } : {},
        attributes: [
          "id",
          "firstname",
          "lastname",
          "address",
          "id_card",
          "profileimage",
        ],
      });
    }

    if (isUserFeatures === "true") {
      usersData = await User.findAll({
        attributes: [
          "id",
          "firstname",
          "lastname",
          "profileimage",
          "height",
          "shoe",
          "clothes",
          "age",
          "parent",
          "professional",
          "exclusive",
          "active",
          "phone",
          "gender",
        ],
        include: [
          {
            model: Feature,
            /*   include: [
              {
                model: FeatureType,
                attributes: ["ftype_color", "ftype_name"],
              },
            ],*/
          },
        ],
        order: [["id", "DESC"]],
      });
    }
    if (isFeatureTypes === "true") {
      ftypes = await FeatureType.findAll();
    }

    if (isFeatures === "true") {
      features = await Feature.findAll({
        //  include: [FeatureType],
        where: userId ? { userId: userId } : {},
        group: ["feat"],
      });
    }

    if (isExtraFeatures === "true") {
      extraFeatures = await ExtraFeats.findAll({
        attributes: ["feat", "id"],
      });
    }

    if (isProfiles === "true") {
      profiles = await Profiles.findAll();
    }

    if (isShortlist === "true") {
      shortlist = await ShortList.findAll();
    }

    if (isNationality === "true") {
      nationality = await Nationality.findAll({
        attributes: ["nationality", "nid"],
      });
    }

    if (isExtraFilters === "true") {
      extrafilters = await ExtraFilters.findAll();
    }

    if (includeGalleries === "true" && userId) {
      gallery = await Image.findAll({
        where: { userId: userId },
      });
    }

    if (includeClients === "true") {
      clients = await Clients.findAll({
        order: [["id", "DESC"]],
      });
    }

    if (includeProjects === "true") {
      projects = await Projects.findAll();
    }

    if (includeProjectTalents === "true") {
      projectTalents = await ProjectsTalents.findAll({
        include: {
          model: User,
          order: [["id", "DESC"]],
          attributes: [
            "id",
            "firstname",
            "lastname",
            "email",
            "address",
            "id_card",
            "parent",
            "profileimage",
            "professional",
          ],
        },
      });
    }

    if (includeProjectUsers === "true") {
      const ids = [];
      await ShortList.findAll().then((shortListData) => {
        shortListData.forEach((item) => {
          try {
            const shortlistItems = JSON.parse(item.shortlist_data);
            const itemIds = shortlistItems.map(
              (shortlistItem) => shortlistItem.id
            );
            ids.push(...itemIds);
          } catch (error) {
            console.error("Error parsing JSON:", error);
          }
        });
      });

      projectUsers = await User.findAll({
        order: [["id", "DESC"]],
        attributes: [
          "id",
          "firstname",
          "lastname",
          "address",
          "id_card",
          "profileimage",
        ],
        where: {
          id: {
            [Op.in]: ids,
          },
        },
      });
    }

    if (includeRoutes === "true") {
      routes = await ClientsRoutes.findAll();
    }

    if (includeClientShare === "true") {
      clientShareData = await ClientShare.findAll();
    }

    // Construct response object with the exact names you want
    const response = {
      onlyUsersData: onlyUsersData,
      usersData: usersData,
      extraFeatures: extraFeatures,
      profiles: profiles,
      shortlist: shortlist,
      nationality: nationality,
      extrafilters: extrafilters,
      features: features,
      ftypes: ftypes,
      gallery: gallery,
      clients: clients,
      projects: projects,
      projectTalents: projectTalents,
      projectUsers: projectUsers,
      routes: routes,
      clientShareData: clientShareData,
      totalUsers: totalUsers,
      usersInProject: usersInProject,
    };

    // Send response with the constructed data
    res.status(200).json(response);
  } catch (error) {
    res.status(400).json({ Error: error.message });
  }
};

/*************************************************************/
/* ---------- LIST EXTRA FEATURES TAGS --------------------- */
/*********************************************************** */
const seqListExtraFeats = async (req, res) => {
  return ExtraFeats.findAll()
    .then((feats) => {
      res.send(feats);
    })
    .catch((error) => {
      console.log(error);
    });
};

/*************************************************************/
/* ---------- FILTER BY TAGS ------------------------------- */
/*********************************************************** */
const seqTags = async (req, res) => {
  /* Associate Tables */
  User.hasMany(Feature);
  Feature.belongsTo(User);
  User.hasMany(Image);
  Image.belongsTo(User);
  User.hasMany(Tags);
  Tags.belongsTo(User);
  /* Set Variables */
  const id = req.params.id;
  const findData = { ...req.body };
  const parseData = findData !== undefined && JSON.parse(findData.tags);
  const extractTags = parseData !== undefined && Object.values(parseData);
  return User.findAll({
    include: [
      {
        model: Feature,
        where: {
          [Op.or]: featFilter,
        },
      },
      { model: Image },
      { model: Tags },
    ],
  })
    .then((response) => res.send(response))
    .catch((error) => {
      console.log("Tag Already Exist");
    });
};

/*************************************************************/
/* ---------- TAGS REMOVE ---------------------------------- */
/*********************************************************** */
const seqRemoveTag = async (req, res) => {
  const id = req.params.id;
  const findData = { ...req.body };
  Tags.destroy({
    where: {
      id: findData.id,
      userId: id,
    },
  }).then(() => res.send("Tag Deleted"));
};

/*************************************************************/
/* ---------- SEARCH USER DATA BY KEYWORD ------------------ */
/*********************************************************** */
const seqFind = async (req, res) => {
  const findData = { ...req.body.data };
  /* Associate Tables */
  User.hasMany(Feature);
  Feature.belongsTo(User);
  User.hasMany(Image);
  Image.belongsTo(User);
  User.hasMany(Tags);
  Tags.belongsTo(User);

  /* Search All Users Data by Keyword */
  return User.findAll({
    include: [{ model: Feature }, { model: Image }, { model: Tags }],
    order: [["id", "DESC"]],
    where: {
      [Op.or]: [
        {
          id: {
            [Op.in]: [`${findData.name}`],
          },
        },
        {
          firstname: {
            [Op.like]: `%${findData.name}%`,
          },
        },
        {
          username: {
            [Op.like]: `%${findData.name}%`,
          },
        },

        {
          lastname: {
            [Op.like]: `%${findData.name}%`,
          },
        },
        {
          age: {
            [Op.like]: `%${findData.name}%`,
          },
        },
        {
          gender: {
            [Op.like]: `%${findData.name}%`,
          },
        },
        {
          height: {
            [Op.like]: `%${findData.name}%`,
          },
        },
        {
          phone: {
            [Op.like]: `%${findData.name}%`,
          },
        },
        {
          email: {
            [Op.like]: `%${findData.name}%`,
          },
        },
        {
          birthyear: {
            [Op.like]: `%${findData.name}%`,
          },
        },
        {
          role: {
            [Op.like]: `%${findData.name}%`,
          },
        },
        {
          active: {
            [Op.like]: `%${findData.name}%`,
          },
        },
      ],
    },
  })
    .then((users) => {
      res.send(users);
    })
    .catch((error) => {
      console.log(error);
    });
};

/*************************************************************/
/* ---------- FILTER FEATURES BY TAGS ---------------------- */
/*********************************************************** */
const seqFilter = (req, res) => {
  /* Associate Tables */
  User.hasMany(Feature);
  Feature.belongsTo(User);
  User.hasMany(Image);
  Image.belongsTo(User);
  User.hasMany(Tags);
  Tags.belongsTo(User);
  /* Set Variables */
  const tags = [...req.body];
  /* Map Tags For Filtering */
  const featFilter = tags.map((tags) => [{ feat: { [Op.like]: `%${tags}%` } }]);
  /* Filter Features By Tags */
  if (featFilter) {
    return User.findAll({
      include: [
        {
          model: Feature,
          where: {
            [Op.or]: featFilter,
          },
        },
        { model: Image },
        { model: Tags },
      ],
    }).then((response) => res.send(response));
  }
};

const sharedClient = async (req, res) => {
  const { pageId } = req.body;
  const token = uuidv4();
  const expiryTime = Date.now() + 24 * 60 * 60 * 1000; // 24 hours expiry
};

const sharedRoutes = async (req, res) => {
  try {
    // Fetch data from multiple tables
    const routes = await ClientsRoutes.findAll();
    // Send response with transformed data

    res.status(200).json({
      routes: encryptData(routes),
    });
  } catch (error) {
    res.status(400).json({ Error: error.message });
  }
};

module.exports = {
  LogOut,
  LogIn,
  seqFind,
  seqFilter,
  seqListAll,
  seqTags,
  seqRemoveTag,
  seqListExtraFeats,
  seqListProjects,
  sharedClient,
  seqSharedList,
  sharedRoutes,
  seqQueryAllData,
};
