const { Sequelize } = require("sequelize");
const sequelize = require("../config/dbseq.js");
const { DataTypes } = Sequelize;

const Projects = sequelize.define(
  "projects",
  {
    project_id: {
      type: DataTypes.INTEGER,
      primaryKey: true,
      autoIncrement: true,
    },
    project_date: {
      type: DataTypes.DATEONLY,
      allowNull: true,
      defaultValue: Sequelize.literal("CURRENT_TIMESTAMP"),
    },
    project_update: {
      type: DataTypes.DATEONLY,
      allowNull: true,
      defaultValue: Sequelize.literal("CURRENT_TIMESTAMP"),
    },
    project_name: {
      type: DataTypes.STRING(500),
      allowNull: true,
      charset: "utf8mb4",
      collate: "utf8mb4_general_ci",
      defaultValue: "NO PROJECT NAME",
      set(value) {
        if (typeof value === "string") {
          this.setDataValue("project_name", value.toLowerCase());
        } else {
          this.setDataValue("project_name", value);
        }
      },
    },
    project_cost: {
      type: DataTypes.INTEGER(10),
      allowNull: true,
      defaultValue: 0,
    },
    project_payed: {
      type: DataTypes.INTEGER(10),
      allowNull: true,
      defaultValue: 0,
    },
    project_due: {
      type: DataTypes.INTEGER(10),
      allowNull: true,
      defaultValue: 0,
    },
    project_client: {
      type: DataTypes.STRING(500),
      allowNull: true,
      charset: "utf8mb4",
      collate: "utf8mb4_general_ci",
      defaultValue: "NO CLIENT",
      set(value) {
        if (typeof value === "string") {
          this.setDataValue("project_client", value.toLowerCase());
        } else {
          this.setDataValue("project_client", value);
        }
      },
    },
    project_client_id: {
      type: DataTypes.STRING(100),
      allowNull: true,
      charset: "utf8mb4",
      collate: "utf8mb4_general_ci",
      defaultValue: "NO ID",
    },
    project_state: {
      type: DataTypes.INTEGER,
      allowNull: true,
      defaultValue: 0,
      validate: {
        min: 0,
        max: 3,
      },
    },
    project_notes: {
      type: DataTypes.STRING(5000),
      allowNull: true,
      charset: "utf8mb4",
      collate: "utf8mb4_general_ci",
    },
    project_talent_users: {
      type: DataTypes.STRING(5000),
      allowNull: true,
      charset: "utf8mb4",
      collate: "utf8mb4_general_ci",
    },
    youtube: {
      type: DataTypes.STRING(1000),
      allowNull: true,
      charset: "utf8mb4",
      collate: "utf8mb4_general_ci",
    },
    currency: {
      type: DataTypes.STRING(10),
      allowNull: true,
      charset: "utf8mb4",
      collate: "utf8mb4_general_ci",
      defaultValue: "RON",
    },
  },
  {
    freezeTableName: true,
    timestamps: false, // If you don't want timestamps like createdAt and updatedAt
  }
);

// Function to update all existing project names and clients to lowercase
async function updateExistingRecordsToLowercase() {
  try {
    // Find all projects
    const projects = await Projects.findAll();

    // Loop through each project and update the fields to lowercase
    for (const project of projects) {
      const updatedProject = {
        project_name: project.project_name
          ? project.project_name.toLowerCase()
          : project.project_name,
        project_client: project.project_client
          ? project.project_client.toLowerCase()
          : project.project_client,
      };

      await project.update(updatedProject);
    }

    console.log(
      "All project names and clients have been updated to lowercase."
    );
  } catch (error) {
    console.error("Error updating project names and clients:", error);
  }
}

// Uncomment the line below to sync the model with the database
// Projects.sync({ alter: true }).then(() => {
//   // Update existing records if necessary
//   if (process.env.UPDATE_EXISTING_RECORDS === 'true') {
//     updateExistingRecordsToLowercase();
//   }
// });
Projects.sync({ alter: true });
module.exports = Projects;
